(*^

::[     frontEndVersion = "Microsoft Windows Mathematica Notebook Front End Version 2.2";
        microsoftWindowsStandardFontEncoding;
        fontset = title, "Times", 24, L0, center, nohscroll, bold;
        fontset = subtitle, "Times", 18, L0, center, nohscroll, bold;
        fontset = subsubtitle, "Times", 14, L0, center, nohscroll, italic;
        fontset = section, "Times", 18, L0, nohscroll, bold, grayBox;
        fontset = subsection, "Times", 14, L0, nohscroll, bold, blackBox;
        fontset = subsubsection, "Times", 12, L0, nohscroll, bold, whiteBox;
        fontset = text, "Times", 12, L0, nohscroll;
        fontset = smalltext, "Times", 10, L0, nohscroll;
        fontset = input, "Courier New", 11, L-5, nowordwrap;
        fontset = output, "Courier New", 11, L-5, nowordwrap;
        fontset = message, "Courier", 12, L-5, nowordwrap, R65280;
        fontset = print, "Courier", 12, L-5, nowordwrap;
        fontset = info, "Courier", 12, L-5, nowordwrap, B65280;
        fontset = postscript, "Courier", 12, L0, nowordwrap;
        fontset = name, "Geneva", 10, L0, nohscroll, italic;
        fontset = header, "Times", 12, L0;
        fontset = footer, "Times", 12, L0, center;
        fontset = help, "Times", 10, L0, nohscroll;
        fontset = clipboard, "Times", 12, L0, nohscroll;
        fontset = completions, "Times", 12, L0, nohscroll;
        fontset = graphics, "Courier New", 10, L0, nowordwrap, nohscroll;
        fontset = special1, "Times", 12, L0, nohscroll;
        fontset = special2, "Times", 12, L0, nohscroll;
        fontset = special3, "Times", 12, L0, nohscroll;
        fontset = special4, "Times", 12, L0, nohscroll;
        fontset = special5, "Times", 12, L0, nohscroll;
        fontset = leftheader, "Times", 12, L2;
        fontset = leftfooter, "Times", 12, L2;
        fontset = reserved1, "Courier New", 10, L0, nowordwrap, nohscroll;]
:[font = smalltext; inactive; preserveAspect; nohscroll; ]
This is version 1.0 of a Mathematica(TM) package which adds features
specific for conformal field theory to the OPEdefs package.
Both are written by
     Kris Thielemans
     Institute for Theoretical Physics
     Celestijnenlaan 200 D
     B-3001 Leuven (Belgium)

     (Email-address is FGBDA40@BLEKUL11.BITNET)

You're free to redistribute this package, on the only condition you
keep this header and don't distribute modified code (well, ask me).

Of course, I do not guarantee you get correct results (even if you use
the package correctly !), although it is tested rather extensively.
This version runs in Mathematica 1.2 and later. I don't expect difficulties
in version 1.1 (but I was not able to check it), but you will have to
change the BeginPackage line, and load OPEdefs manually.

Please contact me if you want to use the package, such that I know who
uses it, and you can get any updates or related packages.  Any
comments, bugs and especially improvements are welcome.

A final note, I almost nowhere test for validity of the arguments you give.
This is done because these checks would take sometimes a (very) long time.
This of course means that you can easily generate nonsense output, when
giving nonsense input.
A common example : use of OPEP[T,A] where T is the Virasoro operator is
invalid (except when c=0) because T is not a primary field.
:[font = smalltext; inactive; nohscroll; ]
Changes in beta 3 (7/6/93) :
- QPNO with a Virasoro operator specified has now syntax QPNO[V,A,B]
- OPEP and OPEQP added
- SetVirasoro remembers centralcharge and doesn't check if V is a Virasoro
  operator when not necessary.
Changes in beta 4 (25/10/93) :
- Primary with several lists as arguments.
- Pblock and QPblock changed to avoid superfluous calls of VirWeight.
- VirWeight for composites of non-primary operators.
Changes in beta 5(25/10/93) :
- added tests such that Sum ranges are always integer in OPEQPPole, and
  removed a bug there.
:[font = input; initialization; preserveAspect; nowordwrap; ]
*)
BeginPackage["OPEconf`", "OPEdefs`"];
(*
:[font = section; inactive; preserveAspect; startGroup; Cclosed; nohscroll; ]
Exported symbols
:[font = input; initialization; preserveAspect; endGroup; nowordwrap; ]
*)
OPEconfVersion = 1.0
Print["OPEconf Version ", OPEconfVersion, " (beta test 6) by Kris Thielemans"]
Print["Type ?OPEconfHelp for a primer on OPEconf."]

OPEconfHelp::usage = StringJoin[
"OPEconf extends the capabilities of OPEdefs when you have a conformal
field theory (with one or more Virasoro operators).\n
You can define your operators and the standard OPEs using Virasoro,
Primary\n
    e.g.: Virasoro[T,c]; Primary[J[i_], 1]; Primary[W,3];\n\n",
"Two new structures aside of OPEData can be used. With MakeOPEP, you
specify only the primaries occuring in the OPE, MakeOPEQP requires a list
of the quasi-primaries. As first parameters to these functions, you have
to give the dimensions of the primary, resp. quasi-primary fields of
which you are giving the OPE. MakeOPE, MakeOPEQP and MakeOPEP can be
to convert a structure to another type.\n
    e.g.: OPE[W,W] = MakeOPE[ MakeOPEP[ 3,2,{c/3 One}] ]\n
OPEPole works also when given a structure returned by MakeOPE(Q)P.
The functions OPE(Q)PPole are completely analogous to OPEPole.\n\n",
"QPNO[A,B] makes the quasi-primary of the same dimension
as NO[A,B], and can be used as an alternative definition of normal
ordering.\n
    e.g.: Virasoro[T,c]; Primary[J,d, 1]; QPNO[T,J]
    --> NO[T,J] - 3/2/(1+2d) J''\n\n",
"WARNING : use only operators with positive dimension.\n
Type ?OPEconf`* for all global symbols"]

OPEQPData::usage =
        "The head of the structure returned by MakeOPEQP. For future
compatibility, do not use its structure directly."

OPEPData::usage =
        "The head of the structure returned by MakeOPEP. For future
compatibility, do not use its structure directly."

 (* test if package loaded first time after OPEdefs. If so, concatenate
    text to usages of MakeOPE and OPEPole
 *)
If[ StringLength[MakeOPE::usage] < 1000,
MakeOPE::usage =
        StringJoin[MakeOPE::usage,
            "\n- A third, and very powerful usage of MakeOPE when OPEconf
is loaded is converting lists of quasi-primaries (OPEQPData) or only the
primaries (OPEPData) into a normal OPE structure.\n
e.g.: Virasoro[T,c];\n
      wwP = MakeOPEP[3,3, {c/3 One}];\n
      ww = OPESimplify[ MakeOPE[wwP], Factor];\n
See also MakeOPEQP, MakeOPEP."
];(* end of StringJoin *)

OPEPole::usage =
        StringJoin[OPEPole::usage,
            "\n\nWhen OPEconf is loaded, OPEPole works also correctly with
an object returned by MakeOPEQP or MakeOPEP.\n
e.g. (W3) : wwP = MakeOPEP[3,3, {c/3 One}];\n
      OPEPole[4][wwP]\n
      --> 32 NO[T, T]/(22 + 5 c) + (-18 + 9 c) T''/(6*(22 + 5*c))\n
When using OPEPole in this way to get a term in the regular part of the
OPE, only descendants of the (quasi-)primaries are computed.  This is
to be contrasted with e.g. OPEPole[-2][A,B] which gives an exact result.\n
See also OPEQPPole, OPEPPole, MakeOPE, MakeOPEQP, MakeOPEP."
] (* end of StringJoin *)
,
 (* second time loaded *)
OPEconf::reload =
        "Warning : OPEconf reloaded without reloading OPEdefs.  This means
that definitions and some stored intermediate results for the OPEs and
composites are not cleared.  If you define new OPEs for the same operators,
this will give wrong results.";

Message[OPEconf::reload]
] (* end of If *)

 (* too long text -> use StringJoin *)
MakeOPEQP::usage = StringJoin[
        "Must be used to make an OPEQPData structure, which represents an
OPE by giving the quasi-primary fields at the different poles. It is assumed
that the fields of which you are giving the OPE are quasi-primary.\n
The functions OPEMap, OPESimplify, MakeOPE, OPEPole are extended to work
with this kind of object. You can also add OPEQPData structures, and
multiply them with a scalar like with normal OPEData's.\n",
"There are three different ways of using MakeOPEP : listing the Ps,
convert to the usual OPE, convert to the listing of the quasi-primaries:\n
- giving it the list of the quasi-primary operators occuring in the poles
of this series. You can give also quasi-primaries occuring in the regular
part of the OPE (see OPEPole). There is no check if the operators listed
are quasi-primaries.\n
    Format:    MakeOPEQP[d1_,d2_, V_, ps_List]\n
    e.g. (for W3) : MakeOPEQP[3,3,{c/3 One}]\n
- converting from an OPEData\n
    Format:    MakeOPEQP[d1_,d2_, V_, ope_OPEData]\n
- converting from an OPEPData, i.e. the listing of the Ps. Currently,
only the singular part is taken into account.\n
    Format:    MakeOPEQP[Ps_OPEPData]\n
(The optional third argument in the two first cases specifies the Virasoro
operator to use. If not given, the current Virasoro operator is used.)\n
See also MakeOPE, MakeOPEP, OPEP, OPEQP."]

MakeOPEP::usage = StringJoin[
        "Must be used to make an OPEPData structure, which represents an
OPE by giving the primary fields at the different poles. It is assumed
that the fields of which you are giving the OPE are primary.\n
The functions OPEMap, OPESimplify, MakeOPE, OPEPole are extended to work
with this kind of object. You can also add OPEPData structures, and
multiply them with a scalar like with normal OPEData's.\n",
"There are three different ways of using MakeOPEP : listing the Ps,
convert to the usual OPE, convert to the listing of the quasi-primaries:\n
- giving it the list of the primary operators occuring in the poles of this
series. You can give also primaries occuring in the regular part of the
OPE (see OPEPole). There is no check if the operators listed are primaries.\n
    Format:    MakeOPEP[d1_,d2_, V_, ps_List]\n
    e.g. (for W3) : MakeOPEP[3,3,{c/3 One}]\n
- converting from an OPEData\n
    Format:    MakeOPEP[d1_,d2_, V_, ope_OPEData]\n
- converting from an OPEQPData, i.e. the listing of the QPs\n
    Format:    MakeOPEP[QPs_OPEQPData]\n
(The optional third argument in the two first cases specifies the Virasoro
operator to use. If not given, the current Virasoro operator is used.)\n
See also MakeOPE, MakeOPEQP, OPEP, OPEQP."]

OPEQP::usage =
     "OPEQP[V,A,B] is equivalent to
MakeOPEQP[dimA,dimB,V,OPE[A,B]]. Here V is the Virasoro
operator you want to use. If V is not specified, the current
Viraosoro operator is used.\n
See also OPEP, MakeOPEQP, SetVirasoro."

OPEP::usage =
     "OPEP[V,A,B] is equivalent to
MakeOPEP[dimA,dimB,V,OPE[A,B]]. Here V is the Virasoro
operator you want to use. If V is not specified, the current
Viraosoro operator is used.\n
See also OPEQP, MakeOPEP, SetVirasoro."

OPEQPPole::usage =
     "OPEQPPole computes the quasi-primary at a pole. You can use it in
four ways (note that the 'V' argument is always optional, and defaults
to the current Virasoro operator) :\n
- OPEQPPole[n][V,A,B]  'A' and 'B' must be quasi-primary, 'n' can be
negative.\n
- OPEQPPole[n][d1, d2, V, ope] with 'd1' and 'd2' the dimensions of the
quasi-primaries which OPE is 'ope', 'n' must be positive.\n
- OPEQPPole[n][ MakeOPEQP[d1,d2, {...}] ]\n
- OPEQPPole[n][ MakeOPEP[d1,d2, {...}] ]\n
See also OPEPole, OPEPPole, QPNO."

OPEPPole::usage =
     "OPEPPole computes the primary at a pole. You can use it in
four ways (note that the 'V' argument is always optional, and defaults
to the current Virasoro operator) :\n
- OPEPPole[n][V,A,B]  'A' and 'B' must be primary, 'n' can be
negative.\n
- OPEPPole[n][d1, d2, V, ope] with 'd1' and 'd2' the dimensions of the
primaries which OPE is 'ope', 'n' must be positive.\n
- OPEPPole[n][ MakeOPEQP[d1,d2, {...}] ]\n
- OPEPPole[n][ MakeOPEP[d1,d2, {...}] ]\n
See also OPEPole, OPEQPPole."

QPNO::usage =
        "QPNO[V,A,B] constructs the quasiprimary (with respect to T)
normal ordered product of A and B in terms of NO[A,B] and other poles of
OPE[A,B]. A and B MUST be quasiprimary. It is equivalent to (but
somewhat faster than) OPEQPPole[0][V, A,B].\n
QPNO[A,B] uses the current Virasoro operator."

Virasoro::usage =
        "Virasoro[T,c] defines T as a bosonic operator which satisfies
the standard Virasoro OPE with central charge c. (See SetVirasoro)"

SetVirasoro::usage =
        "SetVirasoro[T] sets the current Virasoro operator used by Primary,
QuasiP and QPNO. It is called when you use Virasoro[], but you can
call it any time you want."

Primary::usage =
        "Primary[P, dim, bos] defines P as a primary field of dimension
dim with respect to the current Virasoro operator you defined before
(See SetVirasoro). It is assumed that P is fermionic if bos is an odd
integer (tested with OddQ), and bosonic otherwise.\n
Primary[P, dim] is equivalent to Primary[P,dim, 2 dim]\n
Primary[{A,dimA},{B,dimB}] is equivalent to\n
    Primary[A,dimA];Primary[B,dimB]\n
Warning : if you use patterns, they have to have a name.\n
e.g.: Virasoro[T,c]; Primary[J[i_], 1]\n
(This performs the necessary declaration, and sets OPE[T,T] and
OPE[T,J[i_]])."

QPCoefficient::usage =
        "QPCoefficient[d1, d2, d3][n] computes the coefficient of the
n-th derivative of a quasi-primary with dimension d3 in an OPE of
a field with dimension d1 with one with dimension d2."

BetaCoefficients::usage =
        "BetaCoefficients[d1, d2, d3, c][n] computes the coefficients of the
'n'-th level Virasoro descendants of a primary P3 with dimension 'd3' in an
OPE of a field P1 (dimension 'd1') with P2 (dimension 'd2'). 'c' is the
central charge used. If 'c' is not given, the central charge of the
current Virasoro operator is used.\n
It returns a list of two lists : first the list of the coefficients,
second the list of the partitions of 'n'.\n
BetaCoefficients[d1, d2, d3, c][0] is normalized to 1. (This is maybe
different as you expect for P1==P1, P3 == One where the conventional
normalization is 'c'/'d3').\n
e.g.: Virasoro[T,c]; BetaCoefficients[d, 1, 1][2]//Factor\n
  --> {{(d*(-16 + c + 8*d + c*d))/(12*(2 + c)),\n
        -(((-3 + d)*d)/(2 + c))}, \n
       {{1, 1}, {2}}}\n
The corresponding descendants are given by respectively (L[-n] denotes
the n-th mode of the Virasoro operator)\n
   { L[-1] L[-1] P3, L[-2] P3 } == { P3'', NO[T, P3] }"

Null
(*
:[font = section; inactive; preserveAspect; startGroup; nohscroll; ]
Implementation
:[font = input; initialization; preserveAspect; nowordwrap; ]
*)
Begin["`Private`"];
(*
:[font = subsection; inactive; preserveAspect; startGroup; nohscroll; ]
Imports of OPEdefs`Private`
:[font = input; initialization; preserveAspect; endGroup; nowordwrap; ]
*)
If[OPEdefsVersion < 3.1, MaxPole = OPEdefs`Private`MaxPole]
OperatorQ = OPEdefs`Private`OperatorQ
formatstring = OPEdefs`Private`formatstring
(*
:[font = subsection; inactive; preserveAspect; startGroup; nohscroll; ]
Extensions of OPE functions
:[font = input; initialization; preserveAspect; endGroup; nowordwrap; ]
*)
Clear[OPEPData, OPEQPData]

OPEMap[f_, ope_OPEQPData, levelspec___] :=
    MapAt[Map[f, #, levelspec] & , ope, {4}]
OPEMapAt[f_, ope_OPEQPData, position_List] :=
    MapAt[MapAt[f, #, position] & , ope, {4}]
OPEMap[f_, ope_OPEPData, levelspec___] :=
    MapAt[Map[f, #, levelspec] & , ope, {4}]
OPEMapAt[f_, ope_OPEPData, position_List] :=
    MapAt[MapAt[f, #, position] & , ope, {4}]
OPESimplify[A_OPEQPData, func___] :=
        OPEMap[PoleSimplify[#,func]& , A]
OPESimplify[A_OPEPData, func___] :=
        OPEMap[PoleSimplify[#,func]& , A]
(*
:[font = subsection; inactive; preserveAspect; startGroup; nohscroll; ]
Functions to manipulate OPE(Q)Pdatas
:[font = input; initialization; preserveAspect; endGroup; nowordwrap; ]
*)
OPEQPData /: n_ * OPEQPData[d1_,d2_,V_,A_List] := OPEQPData[d1,d2,V,n* A]
OPEPData  /: n_ * OPEPData[d1_,d2_,V_,A_List]  := OPEPData[d1,d2,V,n* A]

extendboth[A_List,start_, min_, max_] :=
        Join[Table[0,{start-min}], A, Table[0,{max+1-start-Length[A]}] ]
OPEQPData[d1_, d2_, V_, A_List] + OPEQPData[d1_, d2_, V_, {}] ^:=
        OPEQPData[d1,d2,V, A]
OPEQPData[d1_, d2_, V_, A_List] + OPEQPData[d1_, d2_, V_, B_List] ^:=
    Block[
        { startdim1 = VirWeight[V][A[[1]]],
          startdim2 = VirWeight[V][B[[1]]],
          mindim = Min[startdim1, startdim2],
          maxdim = Max[startdim1+Length[A], startdim2+Length[B]]-1,
        },
        OPEQPData[d1,d2,V,
            extendboth[A,startdim1, mindim, maxdim] +
            extendboth[B,startdim2, mindim, maxdim]
       ]
    ]

OPEPData[d1_, d2_, V_, A_List] + OPEPData[d1_, d2_, V_, {}] ^:=
        OPEPData[d1,d2,V, A]
OPEPData[d1_, d2_, V_, A_List] + OPEPData[d1_, d2_, V_, B_List]^:=
    Block[
        { startdim1 = VirWeight[V][A[[1]]],
          startdim2 = VirWeight[V][B[[1]]],
          mindim = Min[startdim1, startdim2],
          maxdim = Max[startdim1+Length[A], startdim2+Length[B]]-1,
        },
        OPEPData[d1,d2,V,
            extendboth[A,startdim1, mindim, maxdim] +
            extendboth[B,startdim2, mindim, maxdim]
       ]
    ]

OPEQPData[d1_,d2_,V_,{0, A___}] := OPEQPData[d1,d2,V,{A}]
OPEPData[d1_,d2_,V_,{0, A___}]  := OPEPData[d1,d2,V,{A}]
(*
:[font = subsection; inactive; preserveAspect; startGroup; Cclosed; nohscroll; ]
Virasoro, (Q)Primary and related functions
:[font = input; initialization; preserveAspect; endGroup; nowordwrap; ]
*)
Clear[CentralCharge, VirasoroQ, Virasoro, SetVirasoro,
        Primary, PrimaryQ, QPPrimaryQ]

VirasoroOperator = NoVirasoro;
CentralCharge[NoVirasoro] :=
    (Message[CentralCharge::noVir]; NoCentralCharge)
CentralCharge::noVir = "Warning : no Virasoro operator currently set, using
NoCentralCharge for the central charge in this compution."
CentralCharge[T_] := CentralCharge[T] =
        2 Together[Coefficient[OPEPole[4][T,T], One]]

VirasoroQ[T_] :=
     Block[{ope},
         OperatorQ[T] &&
         ( MatchQ[
             ope = OPESimplify[OPE[T,T] - MakeOPE[{2 T,T'}], Factor],
             OPEData[{_ One, 0,0,0}]
           ] ||
           MatchQ[ ope,
                MakeOPE[{}]
           ] ||
           MatchQ[ ope,
                0
           ]
         )
    ]

SetVirasoro[T_?OperatorQ] :=
    If [VirasoroQ[T],
        VirasoroOperator = T;
        VirasoroQ[T] = True;
        ConfFieldQ[T,T] = True;
        CentralCharge[T],
        (* else *)
        Message[SetVirasoro::noVir, T, VirasoroOperator];$Failed
    ]

SetVirasoro::noVir = "Error : `` does not satisfy the correct OPE.
Current Virasoro operator `` not changed."

Virasoro[T_,c_,one_:One] :=
    Block[{newT = RemovePatterns[T], newc = RemovePatterns[c]},
        Bosonic[T];
        If[SameQ[newT, $Failed] || SameQ[newc, $Failed],
             Message[Virasoro::noname, T,c]; $Failed,
            (* else*)
            If[ !SameQ[T, newT],
                Message[Virasoro::pattern,T,c,VirasoroOperator],
                VirasoroOperator = T
            ];
            CentralCharge[T] = newc;
            VirWeight[T][T] = 2;
            ConfFieldQ[T,T] = True;
            OPE[T,T] = MakeOPE[{newc/2 one, 0, 2 newT, newT'}]
         ]
    ]

Virasoro::noname = "Warning: you should use named patterns in
Virasoro[`1`,`2`]. \n
`1` is declared as an operator, but no OPE is defined.";

Virasoro::pattern = "Warning : current Virasoro operator can not be set
to `1` with central charge `2` due to patterns in `1`.
The current Virasoro operator is `3`. Use SetVirasoro to change this.";

Literal[PrimaryQ[P_, V_:VirasoroOperator]] :=
    Block[{ope},
        OperatorQ[P] &&
        ( MatchQ[
              ope = OPESimplify[OPE[V,P] - MakeOPE[{VirWeight[V][P] P,P'}],
                    Factor],
              OPEData[{}]
          ] ||
          MatchQ[ ope,
               0
          ]
        )
    ]

Primary[a:(_List)..] := Apply[Primary,{a},1]
Primary[P_, dim_, bos_] := (
    If[ NumberQ[bos],
        If[OddQ[bos], Fermionic[P], Bosonic[P]],
        Bosonic[P]; Message[Primary::bosNAN, P, dim, bos]
    ];
    If [SameQ[VirasoroOperator, NoVirasoro],
        Message[Primary::noVir, P],
        Block[{newP = RemovePatterns[P], newdim = RemovePatterns[dim]},
            If[SameQ[newP, $Failed] || SameQ[newdim, $Failed],
                Message[Primary::noname, P, dim]; $Failed,
                (* else*)
                VirWeight[VirasoroOperator][P] = newdim;
                ConfFieldQ[P,VirasoroOperator] = True;
                OPE[VirasoroOperator, P] = MakeOPE[{newdim newP, newP'}]
            ]
        ]
    ]
)
Primary[P_, dim_] := Primary[P, dim, 2 dim]

Primary::noVir = "Warning: you should use Virasoro[] or SetVirasoro[] before
using Primary[`1`,...]. \n
`1` is declared as an operator, but no OPE is defined.";

Primary::bosNAN = "Warning: third argument of Primary[``,``,``] is not
an integer, assuming you want to declare a bosonic operator.";

Primary::noname = "Warning: you should use named patterns in
Primary[`1`,`2`,...]. \n
`1` is declared as an operator, but no OPE is defined.";

Literal[QPrimaryQ[P_, V_:VirasoroOperator]] :=
     Block[{ope},
         OperatorQ[P] &&
         ( MatchQ[
               ope = OPESimplify[OPE[V,P] - MakeOPE[{VirWeight[V][P] P, P'}],
                                 Factor],
               OPEData[{__, 0, 0 ,0}]
           ] ||
           MatchQ[ ope,
                MakeOPE[{}]
           ] ||
           MatchQ[ ope,
                0
           ]
          )
     ]

(*
:[font = subsection; inactive; preserveAspect; startGroup; Cclosed; nohscroll; ]
Auxiliary functions : Max(Q)PPole, VirWeight, RemovePatterns
:[font = subsubsection; inactive; preserveAspect; startGroup; nohscroll; ]
Max(Q)PPole
:[font = input; initialization; preserveAspect; endGroup; nowordwrap; ]
*)
 (* Max(Q)PPole[ope(Q)P] gives the order of the highest pole *)

MaxQPPole::other = "`1` should be a OPEQPData object, something wrong
here, assuming there are no poles in its OPE."
MaxPPole::other = "`1` should be a OPEPData object, something wrong
here, assuming there are no poles in its OPE."

MaxQPPole[OPEQPData[d1_,d2_,V_,{}]] = 0
MaxQPPole[OPEQPData[d1_,d2_,V_,ps_List]] := d1+d2-VirWeight[V][ ps[[1]] ]
MaxQPPole[A_] := (Message[MaxQPPole::other, A]; 0)

MaxPPole[OPEPData[d1_,d2_,V_,{}]] = 0
MaxPPole[OPEPData[d1_,d2_,V_,ps_List]] := d1+d2-VirWeight[V][ ps[[1]] ]
MaxPPole[A_] := (Message[MaxPPole::other, A]; 0)
(*
:[font = input; initialization; nowordwrap; ]
*)
Clear[ConfFieldQ]
Literal[ConfFieldQ[s_ A_,V_]] :=
    ConfFieldQ[A,V] /; OperatorQ[A]
(*Literal[ConfFieldQ[A_Plus,V_]] :=
    True /; And@@ Map[ConfFieldQ[#,V]&, List@@A]*)
Literal[ConfFieldQ[Derivative[i_][A_],V_]] :=
    ConfFieldQ[A,V]
Literal[ConfFieldQ[NO[A_,B_],V_]] :=
    True /; ConfFieldQ[A,V] && ConfFieldQ[B,V]
(*
:[font = subsubsection; inactive; preserveAspect; startGroup; nohscroll; ]
Virweight
:[font = input; initialization; preserveAspect; endGroup; nowordwrap; ]
*)
 (* VirWeight[V_][A_]  computes the Virasoro weight of the
    operator A with respect to the Virasoro operator V
 *)
Clear[VirWeight]
VirWeight[_][0] = 0
 (* VirWeight for composites.
    Note the correction on the usual sum rule. It is
    necessary when OPEPole[1][V,A] is not a derivative.
    I use the coefficient of NO[A,B] in the result. In fact,
    VirWeight should be reimplemented in this case.
 *)
Literal[VirWeight[V_][NO[A_,B_]]] :=
    VirWeight[V][A]+VirWeight[V][B]+
    Coefficient[OPESimplify[OPEPole[1][OPEPole[1][V,A],B]],
        NO[A,B]
    ]//Together

Literal[VirWeight[V_][s_ A_]] :=
    VirWeight[V][A] /; OperatorQ[A]
Literal[VirWeight[V_][Derivative[i_][A_]]] :=
    i + VirWeight[V][A]

Literal[VirWeight[V_][A_+B_]] :=
    VirWeight[V][A] /; ConfFieldQ[A,V]

Literal[VirWeight[V_][A_Plus]] :=
    Block[{sol1, sol2, d, expr = OPESimplify[A,Factor],
        poles = {}, pole1,pole2, operator, coef},
        sol1 = d; Off[Solve::svars];
        If[!SameQ[Head[expr],Plus] || ConfFieldQ[expr,V],
            Return[VirWeight[V][expr]]
        ];
        Do[
            {pole1,pole2} = OPESimplify[
                {#'-OPEPole[1][V,#],d # -OPEPole[2][V,#]},
                Together]& [expr[[i]]];
            If[ SameQ[pole1,0],
                (* make sure d expr[[i]] is a product, such
                   that Select works fine
                *)
                operator = Select[OperatorQ, d expr[[i]], 1]/d;
                If[ SameQ[pole2/.operator->0,0],
                    sol1 = d /. Solve[pole2==0,d][[1]]//Factor;
                    makeConfField[operator,V,sol1];
                    Break[]
                ]
            ];
            AppendTo[poles, {pole1,pole2}]
            ,{i,Length[expr]}
        ];
        If[!SameQ[sol1,d], Return[sol1]];
        var=Array[coef,{Length[expr]}];
        eq = GetCoefficients/@(var . poles)//Together//Numerator;
        sol1 = Map[Together,Solve[eq[[1]]==0,var],{3}];
        If[Length[sol1]==0,
            Message[VirWeight::notConf,A]; Return[0],
        ];
        sol1=First[sol1];
        eq = eq[[2]] /. First[sol1]//Together//Numerator;
        Print["Solving for VirWeight ", Short[A]];
        sol2 = d /. Solve[eq==0/.coef[_]->1,d]//Factor;
        If[Length[sol2]==0,
            Message[VirWeight::notConf,A]; Return[0]
        ];
        d = First[sol2];
        Print["Solved : ", d];
        makeConfField[expr,V,d];
        var2 = freeVar[sol1, var];
        sol2 = Map[Together,First[Solve[eq==0,var]],{2}];
        expr = (List@@expr) . Together[var/.sol1/.sol2];
        (makeConfField[Map[Factor,
                expr /. #->1 /. coef[_]->0],V,d])& /@
            freeVar[sol2, var2];
        On[Solve::svars];
        d
    ]
VirWeight::notConf = "Error : `` is not a conformal field of a
fixed dimension. Using 0 as a result, but abort this calculation.";
makeConfField[operator_,V_,d_] := (
        Print[Short[operator]];
        ConfFieldQ[_ #& /@ operator,V] = True;
        VirWeight[V][_ #& /@ operator] = d;)
makeConfField[operator_Plus,V_,d_] := (
        Print[Short[operator]];
        ConfFieldQ[operator,V] = True;
        VirWeight[V][operator] = d;)
freeVar[sol_List,var_List] :=
        Complement[var, First/@sol]
VirWeight[NoVirasoro][A_] := (Message[VirWeight::noVir, A]; 0)
VirWeight[_][A_] :=
    (Message[VirWeight::noOp, Short[A,2]]; 0) /; !OperatorQ[A]

Literal[VirWeight[V_][A_]] := VirWeight[V][A] =
    Coefficient[OPEPole[2][OPE[V, A]], A]

VirWeight::noVir = "Warning: you should use Virasoro[] or SetVirasoro first.
Assuming `` has weight 0.";
VirWeight::noOp = "Warning: `` is not an Operator, something wrong.
Assuming it has weight 0.";
(*
:[font = subsubsection; inactive; preserveAspect; startGroup; nohscroll; ]
RemovePatterns
:[font = input; initialization; preserveAspect; endGroup; endGroup; nowordwrap; ]
*)
 (* RemovePatterns[x_] removes all (?) patternconstructs from x and
    checks if the pattern has a name. If so, it returns the name,
    otherwise, it returns $Failed.
    RemovePatterns is used by Virasoro and Primary.

    The job to do is rather tricky. The main observation is that the
    pattern a_ or a:(SomePattern) is in FullForm Pattern[a, ...].
    To extract the name a, we have to replace Pattern by a function
    which returns its first argument. Note that this can not be done
    by using pattern matching !
 *)
RemovePatterns[x_] :=
    Block[{ newx},
        newx = x //. {
                RepeatedNull -> (#1&),
                Optional -> (#1&),
                PatternTest -> (#1&),
                Pattern -> (#1&)};
        If[ !FreeQ[newx, Blank] ||
            !FreeQ[newx, BlankNullSequence] ||
            !FreeQ[newx, BlankSequence] ||
            !FreeQ[newx, Alternatives],
            Message[RemovePatterns::noname,x];$Failed,
            (*else*)
            newx
        ]
    ]
RemovePatterns::noname = "`` contains a pattern without a name.";
(*
:[font = subsection; inactive; preserveAspect; startGroup; Cclosed; nohscroll; ]
Coefficients in conformal blocks
:[font = subsubsection; inactive; preserveAspect; startGroup; nohscroll; ]
coefficient of descendant of a QP
:[font = input; initialization; preserveAspect; endGroup; nowordwrap; ]
*)
Clear[QPCoefficient]

QPCoefficient[d1_,d1_,0][0] = 1
QPCoefficient[_,_,0][_] = 0
QPCoefficient[d1_,d2_,d3_][n_]:=
        Pochhammer[d3+d1-d2,n]/Pochhammer[2d3,n]/n!
(*
:[font = subsubsection; inactive; preserveAspect; startGroup; nohscroll; ]
coefficient of descendant of a Primary
:[font = input; initialization; preserveAspect; endGroup; endGroup; nowordwrap; ]
*)
Clear[IntPartitions, IntPartitions0, VirMatrixElement, Sapovalov,
      BetaF, BetaCoefficientsHelp, BetaCoefficients]
 (* IntPartitions[n_,m_] computes the list of all partitions of n
    (positive numbers that sum up to n), the largest number in a partition
    is m. Each partition is ordered from large to small.
 *)
IntPartitions[n_] := IntPartitions[n,n]

IntPartitions[1,_]={{1}}
IntPartitions[n_,large_] := Block[{i},
    Table[
        If[i==n,
            {n},
            Sequence @@ (Prepend[#,i]& /@
                         IntPartitions[n-i, Min[large,i,n-i]])
        ],
    {i,large}]
]
 (* IntPartitions0 computes the partitions without the number 1 *)
IntPartitions0[n_] := Select[IntPartitions[n],FreeQ[#,1]&]

 (* VirMatrixElement[dim, ks, centralcharge] computes the matrix element of
    the list of modes of the Virasoro operator given by "ks". "dim" is the
    dimension of the (primary) state for which the matrix element is computed.
    eg.: VirMatrixElement[dim, {-1, -3, 2, 2}, c]
    How ?
    a dummy head "nc" (for noncommutative) is used to make the list of L's :
        nc[ -1, -3, 2, 2 ]
    A set of rules is applied on this expression until no changes occur :
       - commute negative modes to the right
       Then check
       - negative modes annihilate the state to the left
       - positive modes annihilate the state to the right
       - L[0] on the state gives the "dim" times the state, using this
         and the commutation rules, you get the third rule.
*)
Literal[VirMatrixElement[dim_, ks_List,
                 centralcharge_:(CentralCharge[VirasoroOperator])] ] :=
    Block[{nc},
        FixedPoint[
            Expand[# /.
                nc[a___,n_,m_,b___] :>
                   ( nc[a,m,n,b] +
                     (n-m) nc[a,n+m,b] +
                     If [n+m == 0, centralcharge/12 n(n^2-1) nc[a,b], 0]
                   ) /; n>0 && m<0
           ] /.
           { nc[n_,___]:> 0 /; n<0,
             nc[___,n_]:> 0 /; n>0,
             nc[a___,0,b___] :> (dim+Plus[a]) nc[a,b]
           }
           &,
           nc @@ ks
        ] /. nc[] -> 1
    ]

 (* Sapovalov[dim, ksList] computes the Sapovalov form for the list of ks,
    i.e. a matrix with elements
    VirMatrixElement[dim, Join[ Reverse[ksList[[i]]], -ksList[[j]] ] , c]
 *)
Sapovalov[dim_, ksList_List, c_] :=
     Block[{i,j,m},
        m = Table[
                If[i<=j,
                    VirMatrixElement[
                        dim,Join[Reverse[#[[i]]], -#[[j]]], c
                    ],
                    0],
                {i,Length[#]},{j,Length[#]}
            ]& @ ksList;
        (* make matrix symmetric *)
        Do[ m[[i,j]] = m[[j,i]],
                {i,2,Length[ksList]},{j,i-1}];
        m
     ]

Sapovalov[0, n_Integer,c_] := Sapovalov[0,IntPartitions0[n],c]
Sapovalov[dim_,n_Integer,c_] := Sapovalov[dim,IntPartitions[n],c]

 (* BetaF are some functions needed in the computation of the beta's *)
BetaF[k_,l_,p_][kl_List] :=
    Block[{i,j, n = Length[kl]},
        Product[ kl[[i]] l + p - k + Sum[kl[[j]], {j,i+1, n}], {i,n}]
    ]

 (* BetaCoefficientsHelp computes the betas from the inverse of the
    Sapovalov form and the BetaF
 *)
BetaCoefficientsHelp[_,_,_,_][{}] = {{},{}}
BetaCoefficientsHelp[d1_,d2_,d3_, c_][part_List] :=
    Block[{res},
        res = Together[
                  Inverse[Sapovalov[d3,part, c]] .
                  (BetaF[d2,d1,d3][#]& /@ part)
              ];
      { (* Factor denominators *)
        MapAt[ Factor, res, Position[res, Power[_, -1]] ] ,
        part
      }
   ]

Literal[BetaCoefficients[d1_,d2_,0,
                 centralcharge_:(CentralCharge[VirasoroOperator])][n_] ] :=
    BetaCoefficientsHelp[d1,d2,0, centralcharge][IntPartitions0[n]]
Literal[BetaCoefficients[d1_,d2_,d3_,
                 centralcharge_:(CentralCharge[VirasoroOperator])][n_] ] :=
    BetaCoefficientsHelp[d1,d2,d3, centralcharge][IntPartitions[n]]

(*
:[font = subsection; inactive; preserveAspect; startGroup; Cclosed; nohscroll; ]
Functions that make conformal blocks
:[font = input; initialization; preserveAspect; endGroup; nowordwrap; ]
*)
Clear[QPblock, VirDescendants, VirDescendantsAll, Pblock]

 (* QPblock[d1,d2,d3,qp,V] computes a "quasi-conformal" block, and returns
    it in OPEData form.
    QPblock[d1,d2,d3,qp,V, maxlevel] computes descendants up to maxlevel.
 *)
Literal[QPblock[_,_,_,0 ,___]] = OPEData[{}]
Literal[QPblock[d1_,d2_,d3_,qp_, V_]] :=
    Block[{i},
        OPEData[
            Table[ QPCoefficient[d1,d2,d3][i] Derivative[i][qp],
                {i,0,d1+d2-d3-1}
            ]
        ]
    ]
Literal[QPblock[d1_,d2_,d3_,qp_, V_, maxlevel_]] :=
    Block[{i},
        OPEData[
            Join[
                Table[ QPCoefficient[d1,d2,d3][i] Derivative[i][qp],
                    {i,0,maxlevel}
                ],
                Table[0,{d1+d2-d3-1 - maxlevel}]
            ]
        ]
    ]

 (* VirDescendants[V, primary, partition]
    computes the Virasoro (V) descendant of 'primary' corresponding to
    'partition'. This is easy :
    L[-n] on a field is defined as OPEPole[2 - n][ V, field]
    We have to apply the L's given in the partition, starting with
    last one. Fold does the job, if we Reverse the partition.
 *)
VirDescendants[_,0, _] = 0
VirDescendant[V_, primary_, part_List] :=
    Fold[OPEPole[2-#2][V,#1]&, primary, Reverse[part]]

 (* VirDescendantsAll[d1,d2,d3, V, primary, level]
    computes the sum of the Virasoro (V) descendants of 'level' of the
    'primary', multiplied with the beta-coefficients
 *)
VirDescendantsAll[_,_,_, _, 0, _] = 0

VirDescendantsAll[d1_,d2_,d3_, V_, p_, level_] := 0 /; level < 0
VirDescendantsAll[d1_,d2_,d3_, V_, p_, 0     ] := p
VirDescendantsAll[d1_,d2_,d3_, V_, p_, level_] :=
    Block[{ betas = BetaCoefficients[d1,d2,d3, CentralCharge[V]][level]},
        betas[[1]] . (VirDescendant[V, p, #]&  /@ betas[[2]])
    ] /; level > 0
 (* Last condition not needed in Mathematica Version 2.0, but bug in
    order of the rules stored in previous versions
 *)

Literal[Pblock[_,_,_,0 ,___]] = OPEData[{}];

Literal[Pblock[d1_,d2_,d3_,p_,V_]] :=
    Block[{i},
        OPEData[
            Table[
                VirDescendantsAll[d1,d2,d3, V, p, i],
                {i,0,d1+d2-d3-1}
            ]
        ]
    ]
Literal[Pblock[d1_,d2_,d3_,p_,V_, maxlevel_]] :=
    Block[{i},
        OPEData[
            Join[
                Table[
                    VirDescendantsAll[d1,d2,d3, V, p, i],
                    {i,0,maxlevel}
                ],
                Table[0,{d1+d2-d3-1 - maxlevel}]
            ]
        ]
    ]

(*
:[font = subsection; inactive; preserveAspect; startGroup; Cclosed; nohscroll; ]
MakeOPE*
:[font = subsubsection; inactive; preserveAspect; startGroup; nohscroll; ]
construction of OPE(Q)PData
:[font = input; initialization; preserveAspect; endGroup; nowordwrap; ]
*)
Clear[MakeOPEQP, MakeOPEP]
Literal[MakeOPEQP[d1_,d2_, V_:VirasoroOperator,{}]] :=
    OPEQPData[d1,d2,V,{}]

Literal[MakeOPEQP[d1_,d2_, V_:VirasoroOperator,l_List]] :=
    Block[{positions, ll, sortedlist, i, max},
        ll = Select[l, !SameQ[#,0] && OperatorQ[#]&];
        positions = Expand[-d1-d2+VirWeight[V] /@ ll];
        If[ !Apply[And, IntegerQ /@ positions],
            Message[MakeOPEQP::noint,
                Part[ll,
                    Flatten[Position[IntegerQ /@ positions, False]]
                ]
            ];
            OPEQPData[d1,d2,V,{}],
            (* else *)
            max = Max[-positions];
            sortedlist = Table[0, {max + Max[positions+1]}];
            Do[ sortedlist[[ positions[[i]]+max+1 ]] += ll[[i]],
                {i, Length[ll]}
            ];
            OPEQPData[d1,d2,V,sortedlist]
        ]
    ]
MakeOPEQP::noint =
        "Error : can not determine the exact position in the OPE of the
following operators : ``. Returning a QP OPE without operators."

 (* All other uses of MakeOPEQP are incorrect. *)
MakeOPEQP[l___] :=
    (Message[
         MakeOPEQP::other,
         Short[ HoldForm[MakeOPEQP[l]] ,2]
     ];
     OPEQPData[{}])
MakeOPEQP::other = "Error : `` is not the required
format. Returning a non-valid object."

Literal[MakeOPEP[d1_,d2_, V_:VirasoroOperator,{}]] :=
    OPEPData[d1,d2,V,{}]

Literal[MakeOPEP[d1_,d2_, V_:VirasoroOperator, l_List]] :=
    Block[{positions, ll, sortedlist, i, max},
        ll = Select[l, !SameQ[#,0] && OperatorQ[#]&];
        positions = Expand[-d1-d2+VirWeight[V] /@ ll];
        If[ !Apply[And, IntegerQ /@ positions],
            Message[MakeOPEP::noint,
                Part[ll,
                    Flatten[Position[IntegerQ /@ positions, False]]
                ]
            ];
            OPEPData[d1,d2,V,{}],
            (* else *)
            max = Max[-positions];
            sortedlist = Table[0, {max + Max[positions+1]}];
            Do[ sortedlist[[ positions[[i]]+max+1 ]] += ll[[i]],
                {i, Length[ll]}
            ];
            OPEPData[d1,d2,V,sortedlist]
        ]
    ]
MakeOPEP::noint =
        "Error : can not determine the exact position in the OPE of the
following operators : ``. Returning a QP OPE without operators."

 (* All other uses of MakeOPEP are incorrect. *)
MakeOPEP[l___] :=
    (Message[
         MakeOPEP::other,
         Short[ HoldForm[MakeOPEQP[l]] ,2]
     ];
     OPEPData[{}])
MakeOPEP::other = "Error : `` is not the required
format. Returning a non-valid object.";

(*
:[font = subsubsection; inactive; preserveAspect; startGroup; nohscroll; ]
conversions between OPE*Data
:[font = input; initialization; preserveAspect; endGroup; endGroup; nowordwrap; ]
*)
MakeOPEQP[ope_OPEQPData] := ope

MakeOPEP[ope_OPEPData] := ope

MakeOPE[ OPEQPData[d1_,d2_,V_,l_List] ] :=
    Plus @@ (QPblock[d1,d2,VirWeight[V][#], #, V]& /@ l)

MakeOPE[ OPEPData[d1_,d2_,V_,l_List] ] :=
    Plus @@ (Pblock[d1,d2,VirWeight[V][#], #, V]& /@ l)

Literal[MakeOPEQP[d1_,d2_, V_:VirasoroOperator, ope_OPEData]] :=
    Block[{ restope = ope, lastqp, qps, i },
        qps = Table[
            lastqp = OPEPole[i][restope];
            restope =
                OPESimplify[
                    restope - QPblock[d1,d2,d1+d2-i, lastqp, V],
                    Together
                ];
            lastqp,
            {i, MaxPole[ope], 1, -1}
        ];
        OPEQPData[d1,d2, V, qps]
    ]

Literal[MakeOPEP[d1_,d2_, V_:VirasoroOperator, ope_OPEData]] :=
    Block[{ restope = ope, lastp, ps, i},
        ps = Table[
            lastp = OPEPole[i][restope];
            restope =
                OPESimplify[
                    restope - Pblock[d1,d2,d1+d2-i, lastp, V],
                    Together
                ];
            lastp,
            {i, MaxPole[ope], 1, -1}
        ];
        OPEPData[d1,d2, V, ps]
    ]

MakeOPEQP[ p:(OPEPData[d1_,d2_, V_, ps_List]) ] :=
    MakeOPEQP[d1, d2, V, MakeOPE[p]]

MakeOPEP[ qp:(OPEQPData[d1_,d2_, V_, qps_List]) ] :=
    MakeOPEP[d1, d2, V, MakeOPE[qp]]

(*
:[font = subsection; inactive; startGroup; Cclosed; nohscroll; ]
OPE*
:[font = input; initialization; preserveAspect; endGroup; nowordwrap; ]
*)
protected= Unprotect[OPEQP,OPEP];

Clear[OPEP,OPEQP]
Literal[OPEQP[ V_:VirasoroOperator,
         A_?OperatorQ, B_?OperatorQ]
] :=
     MakeOPEQP[VirWeight[V][A],VirWeight[V][B],V,OPE[A,B]]

Literal[OPEP[ V_:VirasoroOperator,
         A_?OperatorQ, B_?OperatorQ]
] :=
     MakeOPEP[VirWeight[V][A],VirWeight[V][B],V,OPE[A,B]]
Protect@@protected;
(*
:[font = subsection; inactive; preserveAspect; startGroup; nohscroll; ]
OPE*Pole
:[font = subsubsection; inactive; preserveAspect; startGroup; Cclosed; nohscroll; ]
OPE(Q)PPole of OPE(Q)PData
:[font = input; initialization; preserveAspect; endGroup; nowordwrap; ]
*)
Clear[OPEPPole, OPEQPPole]

Literal[OPEQPPole[n_][ opeqp:(OPEQPData[d1_,d2_,V_,qps_List]) ]] :=
    Block[{index},
        index = MaxQPPole[opeqp] - n + 1;
        If [1 <= index <= Length[qps],
             qps[[ index ]],
             0
        ]
    ]

Literal[OPEPPole[i_][ opep:(OPEPData[d1_,d2_,V_,ps_List]) ]] :=
    Block[{index},
        index = MaxPPole[opep] - i + 1;
        If [1 <= index <= Length[ps],
             ps[[ index ]],
             0
        ]
    ]
(*
:[font = subsubsection; inactive; preserveAspect; startGroup; nohscroll; ]
OPEQPPole of 2 operators
:[font = input; initialization; preserveAspect; endGroup; nowordwrap; ]
*)
 (* OPEQPPole[n][A,B]
    extracts the quasi-primary at a certain pole of the OPE of A with B.
    In fact very simple, but implementation is complicated for better
    performance :
    - when n > (a+b)/2, use OPEPole[j][A,B] to compute poles
    - otherwise compute AB = OPE[A,B] first, and use OPEPole[j][AB]
    Note that the (a+b)/2 is a (probably bad) guess for optimum performance
 *)
 (* HighLow optimizations possible :
    compute OPE[A,B,HighPole->2a-1, LowPole->n]
 *)
 (* 1.0 beta 5 (06/01/94)
    added tests for integer sumranges.
    removed bug in the case where the condition in the first If was True.
    to do: change to Pochhammers
 *)
Literal[OPEQPPole[n_][ V_:VirasoroOperator, A_?OperatorQ, B_] ] :=
     Block[{j, tmp, AB, maxAB,
            a = VirWeight[V][A],
            b = VirWeight[V][B],
            fac = 1, dim},
         dim = a + b - n//Expand;
         PoleSimplify[
             If[ IntegerQ[Min[2a - 1, a + b]] &&
                 (!IntegerQ[tmp=Expand[n-Min[2a - 1, a + b]/2]] || tmp>0),
                 OPEPole[n][A,B] +
                 Sum[ ( fac *= -(2a-j)/(j-n)/(2dim-j+n-1) ) *
                      Derivative[j-n][OPEPole[j][A,B]],
                      {j,1+n, Min[2a - 1, a + b]}
                 ],
                 (* else *)
                 AB = OPE[A,B];
                 maxAB = MaxPole[AB];
                 If[ n <= 0,
                     OPEPole[n][A,B] +
                     Sum[ ( fac *= -(2a-n-j)/j/(2dim-j-1) ) *
                          Derivative[j][OPEPole[j+n][A,B]],
                          {j,-n}
                     ],
                     OPEPole[n][AB]
                 ] + (* warning : shift in j *)
                 Sum[ ( fac *= -(2a-j)/(j-n)/(2dim-j+n-1) ) *
                     Derivative[j-n][OPEPole[j][AB]],
                      {j,Max[1,1+n],
                         If[IntegerQ[a]&& (2a-n>0),
                              Min[2a - 1, maxAB],
                              maxAB
                         ]
                      }
                 ]
             ]
         , Together]
     ]
(*
:[font = subsubsection; inactive; preserveAspect; startGroup; Cclosed; nohscroll; ]
OPEQPPole of OPEData, OPEPData
:[font = input; initialization; preserveAspect; endGroup; nowordwrap; ]
*)
 (* 1.0 beta 4 (8/11/93) :
    Removed a problem occuring in OPEQPPole[d1+d2-1][ope].
    In this case the derivative of a dim. zero field (could)
    occur at this pole. This is however a null field, and
    the usual formula breaks down.
 *)
OPEQPPole::NullField =
"Warning : derivative of `1`-th order pole of `2` is a
null-field and can occur with any coefficient in the result
of OPEQPPole. I take this coefficient equal to 0."

Literal[OPEQPPole[n_Integer][d1_,d2_, V_:VirasoroOperator, ope_OPEData]] :=
    Block[{j, fac = 1, dim = d1 + d2 - n//Expand, tmp},
        If [n <= 0, Return[0]];
        If [dim==1,
            tmp = PoleSimplify[OPEPole[d1+d2][ope]',Together];
            If[ !SameQ[tmp,0],
                Message[OPEQPPole::NullField, n,ope]
            ];
            Return[OPEPole[n][ope]]
        ];
        PoleSimplify[
            OPEPole[n][ope] +
            Sum[ Derivative[j-n][OPEPole[j][ope]] *
                 (fac *= -(2 d1 - j)/(j-n)/(2dim-j+n-1)),
                 {j,1+n,MaxPole[ope]}
            ]
        , Together]
    ]

Literal[OPEQPPole[n_][ opep:OPEPData[d1_,d2_, V_, ps_List] ]] :=
    Block[{ restope = OPEData[{}], lastqp = 0},
        Do[
            restope =
                OPESimplify[
                    restope +
                    Pblock[d1,d2, d1+d2-i, OPEPPole[i][opep], V, i-n] -
                    QPblock[d1,d2,d1+d2-i-1, lastqp, V, i-n+1],
                    Together
                ];
            lastqp = OPEPole[i][restope],
            {i, MaxPPole[opep], n, -1}
        ];
        lastqp
    ]
(*
:[font = subsubsection; inactive; preserveAspect; startGroup; Cclosed; nohscroll; ]
OPEPPole of 2 operators
:[font = input; initialization; preserveAspect; endGroup; nowordwrap; ]
*)
 (* test A_?OperatorQ needed to distinguish this type of arguments and the
    arguments for the rule with OPEData
 *)
 (* case of negative 'n' *)
Literal[OPEPPole[n_][ V_:VirasoroOperator, A_?OperatorQ, B_] ] :=
    Block[{i, j,
           d1 = VirWeight[V][A],
           d2 = VirWeight[V][B],
           AB, maxAB, ps},
        ps = MakeOPEP[d1,d2,V, AB = OPE[A,B] ][[-1]];
        maxAB = MaxPole[AB];
        ps = Join[ps, Table[0, {maxAB - Length[ps]}]];
        Do[
            AppendTo[
                ps,
                PoleSimplify[
                    OPEPole[i][A,B] -
                    Sum[
                        VirDescendantsAll[
                            d1,d2,d1+d2-maxAB+j-1, V, ps[[j]], maxAB-j-i+1
                        ],
                        {j, Length[ps]}
                    ],
                    Together
                ]
            ],
            {i, 0, n, -1}
        ];
        If [ Length[ps] == 0, 0, ps[[-1]] ]
    ] /; n<=0
(* !! Warning : assumes no negative dimension fields !! *)
Literal[OPEPPole[n_][ V_:VirasoroOperator, A_?OperatorQ, B_] ] :=
    Block[{i,tmp,
           d1 = VirWeight[V][A],
           d2 = VirWeight[V][B],
           ope, restope, lastp = 0},
        restope = ope =
            If[ !IntegerQ[tmp=Expand[n-(d1+d2)/2]] || tmp>0,
                OPEData[
                    Join[ Table[OPEPole[i][A,B], {i, d1+d2, n, -1}],
                          Table[0              , {i, n-1, 1, -1}]
                    ]
                ],
                OPE[A,B]
            ];
        (* substract Pblock[OPEPole[i+1][restope]]*)
        Do[
            restope =
                OPESimplify[
                    restope - Pblock[d1,d2,d1+d2-i-1, lastp, V, i-n+1],
                    Together
                ];
            lastp = OPEPole[i][restope],
            {i, MaxPole[ope], n, -1}
        ];
        lastp
    ]
(*
:[font = subsubsection; inactive; preserveAspect; startGroup; Cclosed; nohscroll; ]
OPEPPole of OPEData, OPEQPData
:[font = input; initialization; preserveAspect; endGroup; nowordwrap; ]
*)
Literal[OPEPPole[n_][d1_,d2_, V_:VirasoroOperator, ope_OPEData]] :=
    Block[{ restope = ope, lastp = 0, i},
        If [n <= 0, Return[0]];
        Do[
            restope =
                OPESimplify[
                    restope - Pblock[d1,d2,d1+d2-i-1, lastp, V, i-n+1],
                    Together
                ];
            lastp = OPEPole[i][restope],
            {i, MaxPole[ope], n, -1}
        ];
        lastp
    ]
Literal[OPEPPole[n_][ opeqp:(OPEQPData[d1_,d2_, V_, qps_List]) ]] :=
    Block[{ restope = OPEData[{}], lastp = 0, i},
        Do[
            restope =
                OPESimplify[
                    restope +
                    QPblock[d1,d2,d1+d2-i, OPEQPPole[i][opeqp], V, i-n] -
                    Pblock[d1,d2,d1+d2-i-1, lastp, V, i-n+1],
                    Together
                ];
            lastp = OPEPole[i][restope],
            {i, MaxQPPole[opeqp], n, -1}
        ];
        lastp
    ]
(*
:[font = subsubsection; inactive; preserveAspect; startGroup; Cclosed; nohscroll; ]
OPEPole of OPE(Q)PData
:[font = input; initialization; preserveAspect; endGroup; endGroup; nowordwrap; ]
*)
OPEPole[i_][ OPEPData[d1_,d2_,V_,primaries_List] ] :=
    Plus @@
        ( VirDescendantsAll[
              d1,d2,VirWeight[V][#], V, #, d1+d2-VirWeight[V][#]-i
          ]& /@ primaries
        )

OPEPole[i_][ OPEQPData[d1_,d2_,V_,qprimaries_List] ] :=
    Block[{d3, level},
        Plus @@
            ( Function[
                  d3 = VirWeight[V][qp];
                  level = d1+d2-d3-i;
                  If[ level >= 0,
                      QPCoefficient[d1,d2,d3][level] Derivative[level][qp],
                      0
                  ]
                 , qp
              ] /@ qprimaries
            )
    ]
(*
:[font = subsection; inactive; preserveAspect; startGroup; Cclosed; nohscroll; ]
QPNO
:[font = input; initialization; preserveAspect; endGroup; nowordwrap; ]
*)
 (* QPNO[V_,A_,B_] constructs the quasi-normal ordered product
    of A and B (Virasoro operator V). It is actually an
    efficient implementation of OPEQPPole[0][V,A,B]
  *)
Clear[QPNO]

QPNO[A_?OperatorQ,B_?OperatorQ] := QPNO[VirasoroOperator,A, B]

QPNO[V_?OperatorQ, A_?OperatorQ,B_?OperatorQ] :=
    NO[A,B] +
    Block[{AB, maxAB, j,
           a = VirWeight[V][A],
           b = VirWeight[V][B],
           fac = 1, dim},
        AB = OPE[A,B];
        maxAB = MaxPole[AB];
        dim = a+b;
        PoleSimplify[
            Sum[ ( fac *= -(2a-j)/j/(2 dim - j - 1) ) *
                 Derivative[j][OPEPole[j][AB]],
                 {j,maxAB}
            ],
            Together
        ]
    ]

 (*
Clear[QuasiP]
Literal[QuasiP[dim_, V_:VirasoroOperator][A_?OperatorQ,B_?OperatorQ] ] :=
     Block[{j, AB, maxAB,
            a = VirWeight[V][A],
            b = VirWeight[V][B],
            fac = 1, q},
         AB = OPE[A,B];
         maxAB = MaxPole[AB];
         q = dim - a - b//Expand;
         PoleSimplify[
             OPEPole[-q][A,B] +
             Sum[ ( fac *= -(2a+q-j)/j/(2dim-j-1) ) *
                  Derivative[j][OPEPole[j-q][A,B]],
                  {j,q}
             ] + (* warning : shift in j *)
             Sum[ ( fac *= -(2a-j)/(j+q)/(2dim-j-q-1) ) *
                  Derivative[j+q][OPEPole[j][AB]],
                  {j,Max[1,1-q],maxAB}
             ]
         , Together]
     ] /; OperatorQ[V]
 *)
(*
:[font = subsection; inactive; preserveAspect; startGroup; Cclosed; nohscroll; ]
output formats
:[font = input; initialization; preserveAspect; endGroup; nowordwrap; ]
*)
Format[OPEQPData[d1_,d2_,V_,{}]] :=
    StringForm[
        "<QP< {``, ``, ``} || empty >QP>",
        d1,d2,Short[V]
    ]
Format[OPEQPData[d1_,d2_,V_,{a__}]] :=
    StringForm[
        StringJoin[
            "<QP< {``, ``, ``} ||",
            formatstring[
                d1+d2-VirWeight[V][{a}[[1]]], Length[{a}]
            ],
            " >QP>"
        ],
        d1,d2,Short[V],
        a
    ]

Format[OPEPData[d1_,d2_,V_,{}]] :=
    StringForm[
        "<P< {``, ``, ``} || empty >P>",
        d1,d2,Short[V]
    ]
Format[OPEPData[d1_,d2_,V_,{a__}]] :=
    StringForm[
        StringJoin[
            "<P< {``, ``, ``} ||",
            formatstring[
                d1+d2-VirWeight[V][{a}[[1]]], Length[{a}]
            ],
            " >P>"
        ],
        d1,d2,Short[V],
        a
    ]
(*
:[font = input; initialization; preserveAspect; endGroup; nowordwrap; ]
*)
End[];
(*
:[font = input; initialization; preserveAspect; nowordwrap; ]
*)
EndPackage[]
(*
:[font = input; nowordwrap; ]

^*)
